<?php

namespace App\Http\Controllers;

use App\Models\PlacaFrecuente;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Validator;

class PlacaFrecuenteController extends Controller
{
    public function index(): JsonResponse
    {
        $placas = PlacaFrecuente::activos()->get();
        return response()->json(['data' => $placas], JsonResponse::HTTP_OK);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'placa' => ['required', 'string', 'max:20', Rule::unique('placas_frecuentes', 'placa')],
            'conductor' => ['nullable', 'string', 'max:255'],
            'empresa' => ['nullable', 'string', 'max:255'],
            'observaciones' => ['nullable', 'string'],
            'activo' => ['boolean'],
        ], [
            'placa.required' => 'La placa es obligatoria.',
            'placa.unique' => 'Ya existe una placa frecuente con este número.',
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $placa = PlacaFrecuente::create($request->all());
        return response()->json(['data' => $placa], JsonResponse::HTTP_CREATED);
    }

    public function show(PlacaFrecuente $placaFrecuente): JsonResponse
    {
        return response()->json(['data' => $placaFrecuente], JsonResponse::HTTP_OK);
    }

    public function update(Request $request, PlacaFrecuente $placaFrecuente): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'placa' => [
                'required',
                'string',
                'max:20',
                Rule::unique('placas_frecuentes', 'placa')->ignore($placaFrecuente->id)
            ],
            'conductor' => ['nullable', 'string', 'max:255'],
            'empresa' => ['nullable', 'string', 'max:255'],
            'observaciones' => ['nullable', 'string'],
            'activo' => ['boolean'],
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], JsonResponse::HTTP_UNPROCESSABLE_ENTITY);
        }

        $placaFrecuente->update($request->all());
        return response()->json(['data' => $placaFrecuente], JsonResponse::HTTP_OK);
    }

    public function destroy(PlacaFrecuente $placaFrecuente): JsonResponse
    {
        $placaFrecuente->update(['activo' => false]);
        return response()->json(null, JsonResponse::HTTP_NO_CONTENT);
    }
}

