<?php

namespace App\Http\Controllers;

use App\Models\Boleto;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ReporteController extends Controller
{
    /**
     * Reporte de salidas con filtros
     */
    public function salidas(Request $request): JsonResponse
    {
        $query = Boleto::with(['usuarioGenerador', 'usuarioValidador', 'tipoCarga', 'destino']);

        // Filtros
        if ($request->has('estatus')) {
            $query->where('estatus', $request->estatus);
        }

        if ($request->has('usuario_generador_id')) {
            $query->where('usuario_generador_id', $request->usuario_generador_id);
        }

        if ($request->has('folio')) {
            $query->where('folio', 'like', '%' . $request->folio . '%');
        }

        if ($request->has('fecha_desde')) {
            $query->whereDate('fecha_generacion', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('fecha_generacion', '<=', $request->fecha_hasta);
        }

        if ($request->has('placa')) {
            $query->where('placa', 'like', '%' . $request->placa . '%');
        }

        $boletos = $query->orderBy('fecha_generacion', 'desc')->paginate($request->per_page ?? 50);

        return response()->json(['data' => $boletos], JsonResponse::HTTP_OK);
    }

    /**
     * Estadísticas generales para dashboard y reportes
     */
    public function estadisticas(Request $request): JsonResponse
    {
        $query = Boleto::query();

        if ($request->has('fecha_desde')) {
            $query->whereDate('fecha_generacion', '>=', $request->fecha_desde);
        }

        if ($request->has('fecha_hasta')) {
            $query->whereDate('fecha_generacion', '<=', $request->fecha_hasta);
        }

        $hoy = Carbon::today()->toDateString();
        $hasta = $request->get('fecha_hasta', $hoy);
        $desde = $request->get('fecha_desde', Carbon::parse($hasta)->subDays(30)->toDateString());

        // Boletos generados hoy (siempre del día actual, sin filtrar por rango)
        $hoyCount = Boleto::whereDate('fecha_generacion', $hoy)->count();
        // Salidas (validados) hoy
        $salidasHoyCount = Boleto::where('estatus', 'utilizado')->whereDate('fecha_validacion', $hoy)->count();

        // Por día: total de boletos generados por fecha (orden ascendente para la gráfica)
        $porDia = (clone $query)
            ->select(DB::raw('DATE(fecha_generacion) as fecha'), DB::raw('count(*) as total'))
            ->groupBy('fecha')
            ->orderBy('fecha', 'asc')
            ->limit(90)
            ->get();

        // Por día de la semana (últimos 7 días): Dom, Lun, Mar, Mié, Jue, Vie, Sáb
        $hace7Dias = Carbon::today()->subDays(7)->toDateString();
        $porDiaSemanaRaw = Boleto::whereDate('fecha_generacion', '>=', $hace7Dias)
            ->whereDate('fecha_generacion', '<=', $hoy)
            ->selectRaw('DAYOFWEEK(fecha_generacion) as dia, count(*) as total')
            ->groupBy('dia')
            ->pluck('total', 'dia')
            ->toArray();
        $porDiaSemana = [];
        for ($i = 1; $i <= 7; $i++) {
            $porDiaSemana[] = (int) ($porDiaSemanaRaw[$i] ?? 0);
        }

        // Salidas (utilizados) por día - últimos 10 días para sparkline
        $hace9Dias = Carbon::today()->subDays(9)->toDateString();
        $salidasPorDia = Boleto::where('estatus', 'utilizado')
            ->whereDate('fecha_validacion', '>=', $hace9Dias)
            ->whereDate('fecha_validacion', '<=', $hoy)
            ->select(DB::raw('DATE(fecha_validacion) as fecha'), DB::raw('count(*) as total'))
            ->groupBy('fecha')
            ->orderBy('fecha', 'asc')
            ->get()
            ->keyBy('fecha');

        // Rellenar los 10 días (hoy + 9 anteriores) para que el frontend tenga siempre 10 puntos
        $salidasPorDiaArray = [];
        for ($i = 9; $i >= 0; $i--) {
            $fecha = Carbon::today()->subDays($i)->toDateString();
            $row = $salidasPorDia->get($fecha);
            $salidasPorDiaArray[] = [
                'fecha' => $fecha,
                'total' => $row ? (is_array($row) ? ($row['total'] ?? 0) : $row->total) : 0,
            ];
        }

        $estadisticas = [
            'total' => $query->count(),
            'pendientes' => (clone $query)->where('estatus', 'pendiente')->count(),
            'utilizados' => (clone $query)->where('estatus', 'utilizado')->count(),
            'cancelados' => (clone $query)->where('estatus', 'cancelado')->count(),
            'hoy' => $hoyCount,
            'salidas_hoy' => $salidasHoyCount,
            'por_dia' => $porDia,
            'por_dia_semana' => $porDiaSemana,
            'salidas_por_dia' => array_column($salidasPorDiaArray, 'total'),
        ];

        return response()->json(['data' => $estadisticas], JsonResponse::HTTP_OK);
    }

    /**
     * Exportar a CSV
     */
    public function exportarCsv(Request $request): JsonResponse
    {
        $query = Boleto::with(['usuarioGenerador', 'usuarioValidador', 'tipoCarga', 'destino']);

        // Aplicar mismos filtros que salidas
        if ($request->has('estatus')) {
            $query->where('estatus', $request->estatus);
        }
        if ($request->has('fecha_desde')) {
            $query->whereDate('fecha_generacion', '>=', $request->fecha_desde);
        }
        if ($request->has('fecha_hasta')) {
            $query->whereDate('fecha_generacion', '<=', $request->fecha_hasta);
        }

        $boletos = $query->orderBy('fecha_generacion', 'desc')->get();

        // Generar CSV
        $csv = "Folio,Placa,Conductor,Tipo Carga,Destino,Estatus,Generado Por,Validado Por,Fecha Generación,Fecha Validación\n";
        
        foreach ($boletos as $boleto) {
            $csv .= sprintf(
                "%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
                $boleto->folio,
                $boleto->placa,
                $boleto->conductor ?? '',
                $boleto->tipoCarga->nombre ?? '',
                $boleto->destino->nombre ?? '',
                $boleto->estatus,
                $boleto->usuarioGenerador->name ?? '',
                $boleto->usuarioValidador->name ?? '',
                $boleto->fecha_generacion,
                $boleto->fecha_validacion ?? ''
            );
        }

        return response()->json([
            'data' => base64_encode($csv),
            'filename' => 'reporte_salidas_' . date('Y-m-d') . '.csv'
        ], JsonResponse::HTTP_OK);
    }
}

